--[[----------------------------------------------------------------------------

HasselbladTetherTask.lua
Tasks for Lightroom sample Tether plugin

--------------------------------------------------------------------------------

ADOBE SYSTEMS INCORPORATED
 Copyright 2007-2008 Adobe Systems Incorporated
 All Rights Reserved.

NOTICE: Adobe permits you to use, modify, and distribute this file in accordance
with the terms of the Adobe license agreement accompanying it. If you have received
this file from a source other than Adobe, then your use, modification, or distribution
of it requires the prior written permission of Adobe.

------------------------------------------------------------------------------]]

local LrLogger = import 'LrLogger'
local LrPathUtils = import 'LrPathUtils'
--local LrTasks = import 'LrTasks'
--local LrFunctionContext = import 'LrFunctionContext'
local LrFileUtils = import "LrFileUtils"


local LrRemoteCommunication = import 'LrRemoteCommunication'

--============================================================================--

local HasselbladTetherTask = {}

-------------------------------------------------------------------------------

local logger = LrLogger 'HasselbladTetherTask'
	-- not enabled by default

--local ddt = import 'LrTableUtils'.debugDumpTable
local launchAndConnectToImageCaptureTetherServer

-------------------------------------------------------------------------------

function HasselbladTetherTask.init( propertyTable, pluginPath, completionCallback )
	-- Attempt to find and launch the tether application
	
	logger:trace( "plugin path: ", pluginPath, " callback: ", completionCallback )
	
	HasselbladTetherTask._connectedToPluginApp = false
	HasselbladTetherTask._sentListDevicesMessage = false
	HasselbladTetherTask._propertyTable = propertyTable
	
	launchAndConnectToImageCaptureTetherServer( pluginPath, completionCallback )
end

-------------------------------------------------------------------------------

function launchAndConnectToImageCaptureTetherServer( pluginPath, completionCallback )
	-- Attempt a connection
	
	local taskPath
	if MAC_ENV then
	
		-- First try our debug location in case the app is there.
		local parent = LrPathUtils.parent( pluginPath )
		local testPath = LrPathUtils.child( parent, "tether-hasselblad.app/Contents/MacOS/tether-hasselblad" )
		if LrFileUtils.exists( testPath ) then
			taskPath = testPath
		end
	
		if not taskPath then
			taskPath = LrPathUtils.child( pluginPath, "Contents/PlugIns/tether-hasselblad.app/Contents/MacOS/tether-hasselblad" )
		end
		
		logger:trace( "launching plugin app with path: ", taskPath )
	else
		taskPath = pluginPath .. "/tether-image-capture.exe"
		
		logger:trace( "launching plugin app with path: ", taskPath )
	end

	LrRemoteCommunication.spawnTaskAndConnect( "tether_hasselblad", taskPath,
		function( serverRef )
			logger:trace( "launchAndConnectToImageCaptureTetherServer - serverRef: ", serverRef )
			HasselbladTetherTask._serverConnection = serverRef
			if completionCallback then completionCallback( serverRef ~= nil ) end
		end )
	
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.shutdown( propertyTable )
	-- Send a message to the server asking it to shut down
	if HasselbladTetherTask._serverConnection then
		logger:trace( "HasselbladTetherTask.shutdown" )
		LrRemoteCommunication.closeNamedConnection( HasselbladTetherTask._serverConnection, "tether_hasselblad" )
	end
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.setCallback( propertyTable, callback )
	HasselbladTetherTask._hostCallback = callback
end

-------------------------------------------------------------------------------

local function getCaptureInfoCallback( captureId, messageType, result )
	local device = HasselbladTetherTask._connectedDevice
	local capture = device.captures[ captureId ]
	
	logger:trace( 'getCaptureInfoCallback: Received capture info, now starting file download.' )
	
	if messageType == "ok" then
		capture.captureInfo = result
	elseif messageType == "error" then
		capture.captureInfoErr = result
	end
	
	HasselbladTetherTask._hostCallback( "fileInfoAvailable", 
						{ captureId = captureId, 
							fileType = LrPathUtils.extension( result.szFileName ),
							fileLength = result.size, } )

	LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "startDownload", { capture_id = captureId }, 
				function( messageType, result ) if messageType == "error" then capture.errorCode = result end end )
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.tetherCallback( propertyTable, messageType, result )
	logger:trace( "HasselbladTetherTask.tetherCallback - message: ", messageType )

	if messageType == "objectWasAdded" then
		if HasselbladTetherTask._hostCallback then
			local device = HasselbladTetherTask._connectedDevice
			if not device then
				logger:error( "HasselbladTetherTask.tetherCallback: Could not locate device." )
				return
			end
			local captureId = result.capture_id
			if not device.captures then
				device.captures = {}
			end
			if not device.captures[ captureId ] then
				device.captures[ captureId ] = {}
			end
			
			logger:tracef( "objectWasAdded - device %s, device.captures = %s, device.captures[%s] = %s", device, device.captures, captureId, device.captures[captureId] )
			
			local capture = device.captures[ captureId ]
			
			-- Immediately request info about the capture

			LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "getCaptureInfo", { capture_id = captureId }, 
						function( messageType, result ) getCaptureInfoCallback( captureId, messageType, result ) end )
			
			capture.requestedInfo = true
			
			logger:trace( "objectWasAdded - captureId = ", captureId )
			
			HasselbladTetherTask._hostCallback( "objectWasAdded", result )
		end
	elseif messageType == "dataIsAvailable" then
	
	elseif messageType == "thumbnailWasDownloaded" then
		if HasselbladTetherTask._hostCallback then
			local device = HasselbladTetherTask._connectedDevice
			if not device then
				logger:error( "HasselbladTetherTask.tetherCallback: Could not locate device." )
				return
			end
			
			if result.thumbnail_data then
					
				local exifToLROrientation = {
					"AB", "BA", "CD", "DC", "CB", "DA", "AD", "BC",
				}
				
				local orientation = result.image_orientation and exifToLROrientation[ result.image_orientation ]
				
				HasselbladTetherTask._hostCallback( "thumbnailAvailable", { captureId = result.capture_id, thumbnailData = result.thumbnail_data, orientation = orientation } )
			end
					
		end

	elseif messageType == "fileDownloadComplete" then
		if HasselbladTetherTask._hostCallback then
			logger:trace( "HasselbladTetherTask.tetherCallback - fileDownloadComplete: ", result.file_path )
			local device = HasselbladTetherTask._connectedDevice
			
			if not device then
				logger:error( "HasselbladTetherTask.tetherCallback: Could not locate device." )
				return
			end
			local captureId = result.capture_id
			local capture = device.captures[ captureId ]
			
			if not capture then
				logger:error( "HasselbladTetherTask.tetherCallback: could not find capture id: ", captureId )
				return
			end
			
			HasselbladTetherTask._hostCallback( "fileDownloadComplete", 
							{ captureId = captureId, 
								filePath = result.file_path, 
								fileType = LrPathUtils.extension( result.file_path ),
								status = "done" } )
		end
	elseif messageType == "deviceAdded" then
		logger:trace( "HasselbladTetherTask.tetherCallback - deviceAdded: ", result.device_id )
		if HasselbladTetherTask._propertyTable.availableDevices then
			local forceUpdate = false
			for i, device in ipairs( HasselbladTetherTask._propertyTable.availableDevices ) do
				if device.device_id == result.device_id then
					forceUpdate = true
				end
			end
			if not forceUpdate then
				table.insert( HasselbladTetherTask._propertyTable.availableDevices, result )
			end
		--	LrTableUtils.debugDumpTable( HasselbladTetherTask._propertyTable.availableDevices )
			logger:trace( "- deviceAdded forceUpdate: ", forceUpdate )
			HasselbladTetherTask._hostCallback( "availableDevicesChanged", forceUpdate )
			
		end
	elseif messageType == "deviceRemoved" then
		logger:trace( "HasselbladTetherTask.tetherCallback - deviceRemoved: ", result.device_id )
		local deviceId = result.device_id
		if deviceId and HasselbladTetherTask._propertyTable.availableDevices then
			for i, v in ipairs( HasselbladTetherTask._propertyTable.availableDevices ) do
				if v.device_id == deviceId then
					table.remove( HasselbladTetherTask._propertyTable.availableDevices, i )
					HasselbladTetherTask._hostCallback( "availableDevicesChanged" )
					break
				end
			end
		end
	elseif messageType == "cameraSettingsChanged" then
		HasselbladTetherTask._hostCallback( "cameraSettingsChanged", result )
	else
		HasselbladTetherTask._hostCallback( messageType, result )
	end
end

-------------------------------------------------------------------------------


function HasselbladTetherTask.pollForMessages( propertyTable )
--	logger:trace( 'HasselbladTetherTask.pollForMessages()' )
	pcall( function()
		local message, params = LrRemoteCommunication.pollForMessage( HasselbladTetherTask._serverConnection )
		
		if message then
			logger:trace( 'HasselbladTetherTask.pollForMessages() - received message: ', message )
			
			HasselbladTetherTask.tetherCallback( propertyTable, message, params )
		end
	end )
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.startDeviceSearch( propertyTable )
	
	logger:trace( "HasselbladTetherTask.startDeviceSearch" )
	
	if HasselbladTetherTask._connectedToPluginApp then return end
	
	if HasselbladTetherTask._serverConnection then
		HasselbladTetherTask._connectedToPluginApp = true
		HasselbladTetherTask._sentListDevicesMessage = false

		logger:trace( 'HasselbladTetherTask.startDeviceSearch - connection established.' )
								
		HasselbladTetherTask.queryDevices( propertyTable )
	end
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.stopDeviceSearch( propertyTable )
	logger:trace( 'HasselbladTetherTask.stopDeviceSearch' )
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.queryDevices( propertyTable )
	logger:trace( 'HasselbladTetherTask.queryDevices' )
	
	-- If we're still not connected, wait until later
	if not HasselbladTetherTask._connectedToPluginApp then return end
	logger:trace( "HasselbladTetherTask.queryDevices : 1" )
	
	local function listDevicesCallback( messageType, result )
	
		logger:trace( "listDevicesCallback called with result: %s", messageType )
	
		if messageType == "ok" then
			HasselbladTetherTask._propertyTable.availableDevices = result and result.devices or {}
			HasselbladTetherTask._hostCallback( "availableDevicesChanged" )
		end
	end
	
	if not HasselbladTetherTask._sentListDevicesMessage then
		logger:trace( 'HasselbladTetherTask.queryDevices: sending message to server with callback:', listDevicesCallback )
		LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "listDevices", {}, listDevicesCallback )
		HasselbladTetherTask._sentListDevicesMessage = true
	end
	
	if HasselbladTetherTask._propertyTable.availableDevices then
		return { status = 'ok' }
	else
		return { status = 'notReady' }
	end
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.connectDevice( propertyTable, deviceId, callback )
	logger:trace( 'HasselbladTetherTask.connectDevice' )
	
	local function openDeviceCallback( messageType, result )
		if messageType == "ok" then
			HasselbladTetherTask._sessionId = result and result._sessionId
			if callback and result and result.camera_name then
				HasselbladTetherTask._connectedDevice.name = result.camera_name
				callback( result.camera_name )
			end
		end
	end
	
	for i,v in ipairs( HasselbladTetherTask._propertyTable.availableDevices ) do
		if v.device_id == deviceId then
			HasselbladTetherTask._connectedDevice = v
		end
	end

	LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "connectDevice", { device_id = deviceId }, openDeviceCallback )
	
	return { status = 'ok' }
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.disconnectDevice( propertyTable, deviceId )
	logger:trace( 'HasselbladTetherTask.disconnectDevice' )
	
	LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "disconnectDevice", { device_id = deviceId }, nil )
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.doCapture( propertyTable, deviceId )
	logger:trace( 'HasselbladTetherTask.doCapture' )
	
	local function doCaptureCallback( messageType, result )
		if messageType == "ok" then
		end
	end
	
	LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "doCapture", { device_id = deviceId }, doCaptureCallback )
	
	return { status = 'ok' }
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.getCaptureInfo( propertyTable, deviceId, captureId )
	logger:trace( 'HasselbladTetherTask.getCaptureInfo' )

	local device = HasselbladTetherTask._connectedDevice
	local capture = device and device.captures[ captureId ]
	
	if not capture then
		logger:errorf( "HasselbladTetherTask.getCaptureInfo: capture table for captureId %s could not be found.", captureId )
		return {
			status = "error",
			errorMsg = "Could not get capture info for capture id: " .. tostring(captureId),
		}
	end
	
	if capture.captureInfo then
		-- We've already gotten the data
		return {
			status = "ok",
			fileType = LrPathUtils.extension( capture.captureInfo.ifil ),
			fileLength = capture.captureInfo.isiz,
		}
	end
	
	if not capture.requestedInfo then
	end
	
	return {
		status = "notReady",
	}

end

-------------------------------------------------------------------------------

function HasselbladTetherTask.startDownload( propertyTable, deviceId, captureId )
	logger:trace( 'HasselbladTetherTask.startDownload' )
	
	local device = HasselbladTetherTask._connectedDevice
	local capture = device and device.captures[ captureId ]
	if capture then
		LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "startDownload", { capture_id = captureId }, 
					function( messageType, result ) if messageType == "error" then capture.errorCode = result end end )
	end
end

-------------------------------------------------------------------------------

function HasselbladTetherTask.getCameraSettings( propertyTable, deviceId, callbackFn )
	logger:trace( 'HasselbladTetherTask.getCameraSettings' )
	
	LrRemoteCommunication.sendMessageToServer( HasselbladTetherTask._serverConnection, "getCameraSettings", { device_id = deviceId }, 
			function( messageType, result ) 
				logger:trace( 'HasselbladTetherTask.getCameraSettings: received result - ', messageType, result )
				
				if messageType == "error" then 
					callbackFn( nil, result.error_code )
				else
					callbackFn( result )
				end
			end )
end

-------------------------------------------------------------------------------

return HasselbladTetherTask







